const { consultarDetallesCharg } = require('./detallesCharg.model')

function normalizarCharg(charg) {
    if (charg === null || charg === undefined) return null;
    return String(charg).trim();
}

async function enriquecerDatosConDetallesLote(datosArray, tokenSAP, RFC_BASE) {

    const MAX_CONCURRENT = 10;
    const datosEnriquecidos = [];

    const cacheCharg = new Map();

    // datosArray ya es un array de objetos { planilla, proceso, movimientos }
    // NO necesitamos hacer entries() porque ya es un array

    for (let i = 0; i < datosArray.length; i += MAX_CONCURRENT) {
        const lote = datosArray.slice(i, i + MAX_CONCURRENT);

        const resultados = await Promise.all(
            lote.map(async (item) => {
                const { planilla, proceso, movimientos } = item;

                // Si MB51 no devolvió nada
                if (!movimientos || movimientos.length === 0) {
                    return { planilla, proceso, movimientos: [] };
                }

                // Obtener CHARG únicos válidos
                const chargsUnicos = [...new Set(
                    movimientos
                        .filter(m => m.BWART === 543)
                        .map(m => normalizarCharg(m.CHARG))
                        .filter(c => c)
                )];

                const detallesPorCharg = {};

                // Consultar datos por CHARG con cache
                for (const charg of chargsUnicos) {
                    try {
                        if (cacheCharg.has(charg)) {
                            detallesPorCharg[charg] = cacheCharg.get(charg);
                        } else {
                            const detalle = await consultarDetallesCharg(charg, tokenSAP, RFC_BASE);
                            cacheCharg.set(charg, detalle);
                            detallesPorCharg[charg] = detalle;
                        }
                    } catch (err) {
                        console.error(`Error obteniendo detalles CHARG ${charg}:`, err.message);
                        detallesPorCharg[charg] = null; // evitar reventar
                    }
                }

                // Mezclar detalles con cada movimiento
                const movimientosEnriquecidos = movimientos.map(mov => {
                    const c = normalizarCharg(mov.CHARG);
                    return {
                        ...mov,
                        CHARG: c,
                        detalle: detallesPorCharg[c] || null
                    };
                });

                return { planilla, proceso, movimientos: movimientosEnriquecidos };
            })
        );

        datosEnriquecidos.push(...resultados);
    }

    return datosEnriquecidos;
}

module.exports = { enriquecerDatosConDetallesLote };