const {
    Temporada,
    Especie,
    Proveedor,
    Variedad,
    Predio,
    ProcesoMaterial,
    ProcesoCalibreTipo,
    ProcesoCalibre,
    ProcesoTipo,
    ProcesoHeader
} = require('../models/index');

async function prepararDatosParaInsercion(datosEnriquecidos, transaction) {
    const headers = [];
    const entradas = [];
    const salidas = [];
    const omitidos = [];

    // ============================================
    // OBTENER DATOS MAESTROS (UNA SOLA VEZ)
    // ============================================
    const temporadaActiva = await Temporada.findOne({
        where: { gen_estado_temporada: '1' },
        attributes: ['gen_idtbl_temporada'],
        transaction
    });

    const especieCereza = await Especie.findOne({
        where: { gen_nombre_especie: 'Cerezas' },
        attributes: ['gen_idtbl_especie'],
        transaction
    });

    if (!temporadaActiva || !especieCereza) {
        throw new Error('No se encontró temporada activa o especie CEREZA');
    }

    // Buscar material CE_GRANEL una sola vez (CRÍTICO)
    const materialGranel = await ProcesoMaterial.findOne({
        where: { gen_nombre_material: 'CE_GRANEL' },
        attributes: ['gen_idtbl_material'],
        transaction
    });

    if (!materialGranel) {
        throw new Error('Material CE_GRANEL no encontrado en tbl_material. Es requerido para procesar entradas y salidas.');
    }

    // ============================================
    // PROCESAR CADA ORDEN DE PROCESO
    // ============================================
    for (const dato of datosEnriquecidos) {
        const { planilla, proceso, movimientos } = dato;
        const motivosAdvertencia = [];

        // ============================================
        // 1. FILTRO: MOVIMIENTOS PROCESABLES
        // - Entradas (101/102) deben ser de A013
        // - Salidas (543) deben ser de A050 o sin almacén (y tener detalle)
        // ============================================
        const movimientosProcesables = movimientos.filter(m => {
            const lgort = String(m.LGORT || '').trim().toUpperCase();
            const bwart = m.BWART;  //Sin convertir a string

            // Entradas: 101/102 del almacén A013
            if (bwart === 101 || bwart === 102) {
                return lgort === 'A013';
            }

            // Salidas: 543 de A050 o sin almacén (y que tengan detalle)
            if (bwart === 543) {
                return (lgort === 'A050' || lgort === '') && m.detalle;
            }

            return false;
        });

        if (movimientosProcesables.length === 0) {
            omitidos.push({
                planilla,
                motivo: 'Sin movimientos válidos (101/102 de A013 o 543 de A050 con detalle)'
            });
            continue;
        }



        // ============================================
        // 2. VALIDAR Y BUSCAR PROVEEDOR (CRÍTICO)
        // ============================================
        const lifnrUnicos = [...new Set(movimientosProcesables.map(m => m.LIFNR).filter(Boolean))];

        if (lifnrUnicos.length === 0) {
            omitidos.push({
                planilla,
                motivo: 'Sin LIFNR (proveedor) informado en movimientos'
            });
            continue;
        }

        if (lifnrUnicos.length > 1) {
            omitidos.push({
                planilla,
                motivo: `Múltiples proveedores en misma orden: ${lifnrUnicos.join(', ')}`
            });
            continue;
        }

        const lifnr = lifnrUnicos[0];
        const proveedor = await Proveedor.findOne({
            where: { gen_rut_proveedor: lifnr },
            attributes: ['gen_idtbl_proveedor'],
            transaction
        });

        if (!proveedor) {
            omitidos.push({
                planilla,
                motivo: `Proveedor ${lifnr} no existe en BD`
            });
            continue;
        }

        // ============================================
        // 3. VALIDAR TIPO DE OPERACIÓN (CRÍTICO)
        // ============================================
        const matnrTipoOp = movimientosProcesables[0].MATNR;

        if (!matnrTipoOp) {
            omitidos.push({
                planilla,
                motivo: 'MATNR (tipo operación) no informado'
            });
            continue;
        }

        const tipoOp = await ProcesoTipo.findOne({
            where: { gen_nombre_tipo_op: matnrTipoOp },
            attributes: ['gen_idtbl_tipo_op'],
            transaction
        });

        if (!tipoOp) {
            omitidos.push({
                planilla,
                motivo: `Tipo de operación '${matnrTipoOp}' no existe en BD`
            });
            continue;
        }

        // ============================================
        // 4. BUSCAR PREDIO POR CSG (de movimientos 543)
        // ============================================
        const movimientos543 = movimientosProcesables.filter(m => m.BWART === 543);

        if (movimientos543.length === 0) {
            omitidos.push({
                planilla,
                motivo: 'Sin movimientos 543 (salidas con calibre)'
            });
            continue;
        }

        let predio = null;
        let csgEncontrado = null;

        for (const mov of movimientos543) {
            const csg = mov.detalle?.csg;

            if (!csg) continue;

            // Buscar predio por CSG
            predio = await Predio.findOne({
                where: { gen_cod_csg: csg },
                attributes: ['gen_idtbl_predio', 'gen_cod_sdp', 'gen_cod_csg'],
                transaction
            });

            if (predio) {
                csgEncontrado = csg;
                break;
            }
        }

        if (!predio) {
            const csgsIntentados = [...new Set(movimientos543
                .map(m => m.detalle?.csg)
                .filter(Boolean))];

            omitidos.push({
                planilla,
                motivo: `Predio no encontrado. CSG intentados: ${csgsIntentados.join(', ')}`
            });
            continue;
        }

        // ============================================
        // 5. BUSCAR VARIEDAD (OPCIONAL - puede ser NULL)
        // ============================================
        const variedadNombre = movimientos543[0].detalle?.variedad;
        let variedad = null;

        if (variedadNombre) {
            variedad = await Variedad.findOne({
                where: { gen_cod_variedad: variedadNombre },
                attributes: ['idtbl_variedad'],
                transaction
            });

            if (!variedad) {
                motivosAdvertencia.push(`Variedad '${variedadNombre}' no encontrada en BD`);
            }
        } else {
            motivosAdvertencia.push('Variedad no informada');
        }

        // ============================================
        // 6. CONSTRUIR HEADER
        // ============================================
        // const referencia = movimientos543[0].detalle?.referencia ||
        //     movimientos543[0].ZREFERENCIA ||
        //     planilla;

        let referencia = null;

        for (const mov of movimientos543) {
            const redDetalle = mov.detalle?.referencia;
            if (redDetalle && redDetalle.trim() !== '' && redDetalle !== planilla) {
                referencia = redDetalle;
                break;
            }
        }

        if (!referencia) {
            for (const mov of movimientos543) {
                const refZRef = mov.ZREFERENCIA;
                if (refZRef && String(refZRef).trim() !== '' && String(refZRef) !== planilla) {
                    referencia = String(refZRef);
                    break;
                }
            }
        }

        // Si aún no encontró referencia válida, usar planilla como fallback
        if (!referencia) {
            referencia = planilla;
        }

        const fechaProceso = movimientos543[0].CPUDT_MKPF?.split('T')[0] ||
            movimientos543[0].BUDAT?.split('T')[0] ||
            new Date().toISOString().split('T')[0];

        const header = {
            gen_fecha_header_op: fechaProceso,
            gen_orden_produccion: planilla,
            gen_referencia_header_op: referencia,
            gen_exp_header_op: null,
            gen_estado_op: '0',
            gen_comentarios_op: null,
            gen_fk_tbl_temporada_gen_idtbl_temporada: temporadaActiva.gen_idtbl_temporada,
            gen_fk_tbl_especie_gen_idtbl_especie: especieCereza.gen_idtbl_especie,
            gen_fk_tbl_variedad_idtbl_variedad: variedad?.idtbl_variedad || null,
            gen_fk_tbl_predio_gen_idtbl_predio: predio.gen_idtbl_predio,
            gen_fk_tbl_tipo_op_idtbl_tipo_op: tipoOp.gen_idtbl_tipo_op,
            gen_fk_tbl_proveedor_idtbl_proveedor: proveedor.gen_idtbl_proveedor,
            gen_fk_tbl_proceso_setup_idtbl_proceso_setup: null,
            extra_1: null,
            extra_2: null,
            extra_3: null
        };

        //Validar Duplicados - Antes de crear el header
        const headerExistente = await ProcesoHeader.findOne({
            where: {
                gen_orden_produccion: planilla,
            },
            transaction
        })

        if (headerExistente) {
            omitidos.push({
                planilla,
                motivo: `Orden ${planilla} ya existe en BD`
            });
            continue;
        }


        headers.push(header);

        // ============================================
        // 7. ENTRADAS (movimientos 101) - SUMA TOTAL
        // ============================================
        const movimientos101 = movimientosProcesables.filter(m => m.BWART === 101);

        if (movimientos101.length === 0) {
            motivosAdvertencia.push('Sin movimientos 101 (entradas)');
        }

        const kilosTotales = movimientos101.reduce((sum, m) =>
            sum + parseFloat(m.ERFMG || 0), 0
        );

        entradas.push({
            planilla,
            gen_cantidad_in_op: kilosTotales,
            gen_fk_tbl_material_idtbl_material: materialGranel.gen_idtbl_material,
            gen_referencia: referencia
        });

        // ============================================
        // 8. SALIDAS - MOVIMIENTOS 543 (SIN AGRUPAR - UN REGISTRO POR MOVIMIENTO)
        // ============================================
        console.log(`\n [${planilla}] Procesando ${movimientos543.length} movimientos 543:`);

        for (const mov of movimientos543) {
            console.log(`   🔍 Mov 543 - CHARG: ${mov.CHARG} - Detalle completo:`, mov.detalle);
            const calibreNombre = mov.detalle?.calibre;
            const categoria = mov.detalle?.categoria;
            const kilos = parseFloat(mov.ERFMG || 0);

            // Saltar si no hay kilos
            if (kilos <= 0) continue;

            // Buscar calibre
            let calibre = null;
            let calibreTipo = null;

            //EXCLUIR DES de la búsqueda de calibre (tiene tratamiento especial)
            if (calibreNombre &&
                calibreNombre !== 'SIN_CALIBRE' &&
                calibreNombre.toUpperCase() !== 'DES') {

                calibre = await ProcesoCalibre.findOne({
                    where: { gen_nombre_calibre: calibreNombre },
                    attributes: ['gen_idtbl_calibre', 'mercado_default'],
                    transaction
                });

                //Buscar relación calibre-tipo
                if (calibre) {
                    calibreTipo = await ProcesoCalibreTipo.findOne({
                        where: { gen_fk_tbl_calibre_gen_idtbl_calibre: calibre.gen_idtbl_calibre },
                        attributes: ['gen_idtbl_calibre_tipo'],
                        transaction
                    });
                }

                if (!calibre) {
                    motivosAdvertencia.push(`Calibre '${calibreNombre}' no encontrado en BD`);
                }
            }

            //Determinar destino
            let destinoOp = calibre?.mercado_default || null;

            //Casos especiales: DES (Descarte)
            if (calibreNombre && calibreNombre.toUpperCase() === "DES") {
                destinoOp = 4;

                //Crear registro especial para DES
                salidas.push({
                    planilla,
                    gen_cantidad_out_op: kilos,
                    gen_categoria_out_op: "SIN_CATEGORIA",
                    gen_fk_tbl_material_idtbl_material: 190,
                    gen_fk_tbl_calibre_tipo_gen_idtbl_calibre_tipo: 209,
                    gen_destino_op: destinoOp,
                    extra_2: null,
                    extra_3: null
                });
            } else {
                //Crear registro normal para otros calibres
                salidas.push({
                    planilla,
                    gen_cantidad_out_op: kilos,
                    gen_categoria_out_op: categoria || null,
                    gen_fk_tbl_material_idtbl_material: materialGranel.gen_idtbl_material,
                    gen_fk_tbl_calibre_tipo_gen_idtbl_calibre_tipo: calibreTipo?.gen_idtbl_calibre_tipo || null,
                    gen_destino_op: destinoOp,
                    extra_2: null,
                    extra_3: null
                });
            }
        }

        //Log de advertencias (FUERA del for de movimientos)
        if (motivosAdvertencia.length > 0) {
            console.warn(`[${planilla}] Procesado con advertencias: ${motivosAdvertencia.join(' | ')}`);
        }
    }

    // ============================================
    // RESUMEN FINAL (FUERA de todos los loops)
    // ============================================
    console.log(`
        RESUMEN PREPARACIÓN DATOS:
        Headers preparados:  ${headers.length}
        Entradas preparadas: ${entradas.length}
        Salidas preparadas:  ${salidas.length}
        Omitidos:            ${omitidos.length}
    `);

    if (omitidos.length > 0) {
        console.warn('ÓRDENES DE PROCESO OMITIDAS:');
        omitidos.forEach(o => console.warn(`   - Orden: ${o.planilla} | Motivo: ${o.motivo}`));
    }

    return { headers, entradas, salidas, omitidos };
}

module.exports = { prepararDatosParaInsercion };